<?php
session_start();
require_once "config/config.php";

// Include library untuk membaca Excel
require_once __DIR__ . "/vendor/autoload.php";
use PhpOffice\PhpSpreadsheet\IOFactory;

class ImportAuditBerkas
{
	private $db;

	public function __construct($pdo)
	{
		$this->db = $pdo;
	}

	/**
	 * Proses import file Excel
	 */
	public function importFromExcel($filePath, $auditId, $userId)
	{
		try {
			// Load file Excel
			$spreadsheet = IOFactory::load($filePath);
			$worksheet = $spreadsheet->getActiveSheet();

			// Ambil data dari worksheet
			$data = [];
			$highestRow = $worksheet->getHighestRow();
			$importCount = 0;
			$skipCount = 0;
			$errorDetails = [];
			$subElemenCount = 0;
			$subSubElemenCount = 0;

			for ($row = 2; $row <= $highestRow; $row++) {
				// Kolom A: NO
				// Kolom B-F: Bisa di-merge, data sebenarnya ada di G
				// Kolom G: SAMPLE / CATATAN KHUSUS (data sebenarnya)

				$no = $worksheet->getCell("A" . $row)->getValue();

				// Coba baca dari kolom G (untuk sample/catatan khusus) terlebih dahulu
				$sample = trim($worksheet->getCell("G" . $row)->getValue() ?? "");

				// Jika kolom G kosong, coba dari F
				if (empty($sample)) {
					$sample = trim($worksheet->getCell("F" . $row)->getValue() ?? "");
				}

				// Untuk kode, coba dari berbagai kolom karena bisa di-merge
				$kodeSubElemen = "";
				$kodeSubSubElemen = "";
				$kriteria = "";
				$metodeAudit = "";

				// Coba baca data dari kolom B-F
				for ($col = 'B'; $col <= 'F'; $col++) {
					$value = trim($worksheet->getCell($col . $row)->getValue() ?? "");
					if (empty($value)) continue;

					// Deteksi apakah ini kode sub elemen
					if ($this->isKodeSubElemen($value)) {
						$kodeSubElemen = $value;
					}
					// Deteksi apakah ini kode sub sub elemen
					elseif ($this->isKodeSubSubElemen($value)) {
						$kodeSubSubElemen = $value;
					}
					// Jika bukan kode, anggap sebagai kriteria
					elseif (empty($kriteria)) {
						$kriteria = $value;
					}
					// Jika kriteria sudah ada, anggap sebagai metode audit
					elseif (empty($metodeAudit)) {
						$metodeAudit = $value;
					}
				}

				// Validasi sample harus ada (ini yang paling penting)
				if (empty($sample)) {
					continue;
				}

				// Log untuk debugging
				error_log("Row $row: Sample='$sample', KodeSub='$kodeSubElemen', KodeSubSub='$kodeSubSubElemen', Kriteria='$kriteria'");

				// Validasi format kode sub elemen
				if (!empty($kodeSubElemen)) {
					$cleanKode = $this->cleanKodeSubElemen($kodeSubElemen);
					if ($cleanKode !== $kodeSubElemen) {
						$kodeSubElemen = $cleanKode;
					}
				}

				// Validasi format kode sub sub elemen
				if (!empty($kodeSubSubElemen)) {
					$cleanKode = $this->cleanKodeSubSubElemen($kodeSubSubElemen);
					if ($cleanKode !== $kodeSubSubElemen) {
						$kodeSubSubElemen = $cleanKode;
					}
				}

				// Deteksi apakah ini sub_sub_elemen atau sub_elemen
				if (!empty($kodeSubSubElemen)) {
					// Ini adalah sub_sub_elemen
					$subSubElemenId = $this->getSubSubElemenId(
						$kodeSubSubElemen,
						$auditId,
					);
					if ($subSubElemenId) {
						if (
							$this->saveSubSubElemenBerkas(
								$subSubElemenId,
								$sample,
								$userId,
							)
						) {
							$importCount++;
							$subSubElemenCount++;
						}
					} else {
						$skipCount++;
						$errorDetails[] = "Baris $row: Sub Sub Elemen dengan kode '$kodeSubSubElemen' tidak ditemukan";
					}
				} elseif (!empty($kodeSubElemen)) {
					// Ini adalah sub_elemen
					$subElemenId = $this->getSubElemenId($kodeSubElemen, $auditId);
					if ($subElemenId) {
						if (
							$this->saveSubElemenBerkas($subElemenId, $sample, $userId)
						) {
							$importCount++;
							$subElemenCount++;
						}
					} else {
						$skipCount++;
						$errorDetails[] = "Baris $row: Sub Elemen dengan kode '$kodeSubElemen' tidak ditemukan";
					}
				} else {
					// Jika tidak ada kode yang terdeteksi, coba cocokkan berdasarkan kriteria
					if (!empty($kriteria)) {
						$matchedId = $this->findElemenByKriteria($kriteria, $auditId);
						if ($matchedId) {
							if ($this->saveSubElemenBerkas($matchedId, $sample, $userId)) {
								$importCount++;
								$subElemenCount++;
								error_log("Row $row: Matched by kriteria '$kriteria' -> ID $matchedId");
							}
						} else {
							// Jika tidak ada yang cocok, simpan sebagai data tanpa kode (untuk review manual)
							$this->saveUnknownData($row, $kriteria, $sample, $userId);
							$importCount++; // Tetap dihitung sebagai import
							error_log("Row $row: Saved as unknown data - '$kriteria'");
						}
					} else {
						// Jika hanya ada sample tanpa kode atau kriteria
						$this->saveUnknownData($row, "Unknown", $sample, $userId);
						$importCount++;
						error_log("Row $row: Saved as unknown data with only sample");
					}
				}
			}

			// Buat summary hasil import
			$message = "Import selesai. $importCount data berhasil diimport";
			if ($skipCount > 0) {
				$message .= ", $skipCount data dilewati";
			}

			return [
				"status" => "success",
				"message" => $message,
				"details" => [
					"imported" => $importCount,
					"skipped" => $skipCount,
					"sub_elemen_count" => $subElemenCount,
					"sub_sub_elemen_count" => $subSubElemenCount,
					"errors" => $errorDetails,
				],
			];
		} catch (Exception $e) {
			return [
				"status" => "error",
				"message" => "Error: " . $e->getMessage(),
			];
		}
	}

	/**
	 * Mengecek apakah string adalah kode sub elemen
	 */
	private function isKodeSubElemen($kode)
	{
		if (empty($kode)) return false;

		// Hilangkan spasi berlebih
		$kode = preg_replace("/\s+/", "", $kode);

		// Jika hanya huruf romawi (I, II, III, IV, V)
		if (preg_match('/^(I|II|III|IV|V|VI|VII|VIII|IX|X)$/', $kode)) {
			return true;
		}

		// Jika format sudah benar (contoh: I.1, II.2, III.3)
		if (preg_match('/^(I|II|III|IV|V|VI|VII|VIII|IX|X)\.\d+$/', $kode)) {
			return true;
		}

		return false;
	}

	/**
	 * Mengecek apakah string adalah kode sub sub elemen
	 */
	private function isKodeSubSubElemen($kode)
	{
		if (empty($kode)) return false;

		// Hilangkan spasi berlebih
		$kode = preg_replace("/\s+/", "", $kode);

		// Jika format sudah benar (contoh: I.1.1, II.2.1)
		if (preg_match('/^(I|II|III|IV|V|VI|VII|VIII|IX|X)\.\d+\.\d+$/', $kode)) {
			return true;
		}

		return false;
	}

	/**
	 * Membersihkan dan format kode sub elemen
	 */
	private function cleanKodeSubElemen($kode)
	{
		// Hilangkan spasi berlebih
		$kode = preg_replace("/\s+/", "", $kode);

		// Jika hanya huruf romawi (I, II, III, IV, V), buat jadi format I
		if (preg_match('/^(I|II|III|IV|V|VI|VII|VIII|IX|X)$/', $kode)) {
			return $kode;
		}

		// Jika format sudah benar (contoh: I.1, II.2, III.3)
		if (preg_match('/^(I|II|III|IV|V|VI|VII|VIII|IX|X)\.\d+$/', $kode)) {
			return $kode;
		}

		return $kode;
	}

	/**
	 * Membersihkan dan format kode sub sub elemen
	 */
	private function cleanKodeSubSubElemen($kode)
	{
		// Hilangkan spasi berlebih
		$kode = preg_replace("/\s+/", "", $kode);

		// Jika format sudah benar (contoh: I.1.1, II.2.1)
		if (preg_match('/^(I|II|III|IV|V|VI|VII|VIII|IX|X)\.\d+\.\d+$/', $kode)) {
			return $kode;
		}

		return $kode;
	}

	/**
	 * Mendapatkan ID sub_elemen berdasarkan kode dan audit
	 */
	private function getSubElemenId($kodeSubElemen, $auditId)
	{
		// Clean kode
		$kodeSubElemen = $this->cleanKodeSubElemen($kodeSubElemen);

		$sql = "SELECT se.id
                FROM sub_elemen se
                JOIN elemen e ON se.elemen_id = e.id
                WHERE se.kode_sub_elemen = :kode_sub_elemen
                AND e.audit_id = :audit_id
                AND se.status = 'aktif'";

		$stmt = $this->db->prepare($sql);
		$stmt->execute([
			":kode_sub_elemen" => $kodeSubElemen,
			":audit_id" => $auditId,
		]);

		$result = $stmt->fetch(PDO::FETCH_ASSOC);
		return $result ? $result["id"] : null;
	}

	/**
	 * Mendapatkan ID sub_sub_elemen berdasarkan kode
	 */
	private function getSubSubElemenId($kodeSubSubElemen, $auditId)
	{
		// Clean kode
		$kodeSubSubElemen = $this->cleanKodeSubSubElemen($kodeSubSubElemen);

		$sql = "SELECT sse.id
                FROM sub_sub_elemen sse
                JOIN sub_elemen se ON sse.sub_elemen_id = se.id
                JOIN elemen e ON se.elemen_id = e.id
                WHERE sse.kode_sub_sub_elemen = :kode_sub_sub_elemen
                AND e.audit_id = :audit_id
                AND sse.status = 'aktif'";

		$stmt = $this->db->prepare($sql);
		$stmt->execute([
			":kode_sub_sub_elemen" => $kodeSubSubElemen,
			":audit_id" => $auditId,
		]);

		$result = $stmt->fetch(PDO::FETCH_ASSOC);
		return $result ? $result["id"] : null;
	}

	/**
	 * Menyimpan data ke tabel sub_elemen_berkas
	 */
	private function saveSubElemenBerkas($subElemenId, $catatanKhusus, $userId)
	{
		// Cek apakah data sudah ada
		$sql = "SELECT id FROM sub_elemen_berkas
                WHERE sub_elemen_id = :sub_elemen_id
                AND catatan_khusus = :catatan_khusus";

		$stmt = $this->db->prepare($sql);
		$stmt->execute([
			":sub_elemen_id" => $subElemenId,
			":catatan_khusus" => $catatanKhusus,
		]);

		if (!$stmt->fetch()) {
			// Insert data baru
			$sql = "INSERT INTO sub_elemen_berkas
                    (sub_elemen_id, jenis_berkas_id, nomor_urut, catatan_khusus, created_by)
                    VALUES
                    (:sub_elemen_id, :jenis_berkas_id, :nomor_urut, :catatan_khusus, :created_by)";

			$stmt = $this->db->prepare($sql);
			$result = $stmt->execute([
				":sub_elemen_id" => $subElemenId,
				":jenis_berkas_id" => 1, // Default jenis berkas
				":nomor_urut" => $this->getNextNomorUrut(
					"sub_elemen_berkas",
					"sub_elemen_id",
					$subElemenId,
				),
				":catatan_khusus" => $catatanKhusus,
				":created_by" => $userId,
			]);

			return $result;
		}

		return true; // Data sudah ada
	}

	/**
	 * Menyimpan data ke tabel sub_sub_elemen_berkas
	 */
	private function saveSubSubElemenBerkas($subSubElemenId, $catatanKhusus, $userId)
	{
		// Cek apakah data sudah ada
		$sql = "SELECT id FROM sub_sub_elemen_berkas
                WHERE sub_sub_elemen_id = :sub_sub_elemen_id
                AND catatan_khusus = :catatan_khusus";

		$stmt = $this->db->prepare($sql);
		$stmt->execute([
			":sub_sub_elemen_id" => $subSubElemenId,
			":catatan_khusus" => $catatanKhusus,
		]);

		if (!$stmt->fetch()) {
			// Insert data baru
			$sql = "INSERT INTO sub_sub_elemen_berkas
                    (sub_sub_elemen_id, jenis_berkas_id, nomor_urut, catatan_khusus, created_by)
                    VALUES
                    (:sub_sub_elemen_id, :jenis_berkas_id, :nomor_urut, :catatan_khusus, :created_by)";

			$stmt = $this->db->prepare($sql);
			$result = $stmt->execute([
				":sub_sub_elemen_id" => $subSubElemenId,
				":jenis_berkas_id" => 1, // Default jenis berkas
				":nomor_urut" => $this->getNextNomorUrut(
					"sub_sub_elemen_berkas",
					"sub_sub_elemen_id",
					$subSubElemenId,
				),
				":catatan_khusus" => $catatanKhusus,
				":created_by" => $userId,
			]);

			return $result;
		}

		return true; // Data sudah ada
	}

	/**
	 * Mendapatkan nomor urut berikutnya
	 */
	private function getNextNomorUrut($table, $column, $id)
	{
		$sql = "SELECT MAX(nomor_urut) as max_urut FROM {$table} WHERE {$column} = :id";
		$stmt = $this->db->prepare($sql);
		$stmt->execute([":id" => $id]);
		$result = $stmt->fetch(PDO::FETCH_ASSOC);

		return $result && $result["max_urut"] ? $result["max_urut"] + 1 : 1;
	}

	/**
	 * Mencocokkan elemen berdasarkan kriteria/nama
	 */
	private function findElemenByKriteria($kriteria, $auditId)
	{
		// Coba cari di sub_elemen berdasarkan nama_sub_elemen
		$sql = "SELECT se.id
                FROM sub_elemen se
                JOIN elemen e ON se.elemen_id = e.id
                WHERE se.nama_sub_elemen LIKE :kriteria
                AND e.audit_id = :audit_id
                AND se.status = 'aktif'
                LIMIT 1";

		$stmt = $this->db->prepare($sql);
		$stmt->execute([
			":kriteria" => "%" . $kriteria . "%",
			":audit_id" => $auditId,
		]);

		$result = $stmt->fetch(PDO::FETCH_ASSOC);
		if ($result) {
			return $result["id"];
		}

		// Jika tidak ketemu, coba di sub_sub_elemen
		$sql = "SELECT sse.id
                FROM sub_sub_elemen sse
                JOIN sub_elemen se ON sse.sub_elemen_id = se.id
                JOIN elemen e ON se.elemen_id = e.id
                WHERE sse.nama_sub_sub_elemen LIKE :kriteria
                AND e.audit_id = :audit_id
                AND sse.status = 'aktif'
                LIMIT 1";

		$stmt = $this->db->prepare($sql);
		$stmt->execute([
			":kriteria" => "%" . $kriteria . "%",
			":audit_id" => $auditId,
		]);

		$result = $stmt->fetch(PDO::FETCH_ASSOC);
		return $result ? $result["id"] : null;
	}

	/**
	 * Menyimpan data yang tidak dikenal untuk review manual
	 */
	private function saveUnknownData($row, $kriteria, $sample, $userId)
	{
		// Buat tabel temporary jika belum ada
		$this->createUnknownDataTable();

		$sql = "INSERT INTO unknown_import_data
                (baris_number, kriteria, sample, created_by, created_at)
                VALUES
                (:baris_number, :kriteria, :sample, :created_by, NOW())";

		$stmt = $this->db->prepare($sql);
		return $stmt->execute([
			":baris_number" => $row,
			":kriteria" => $kriteria,
			":sample" => $sample,
			":created_by" => $userId,
		]);
	}

	/**
	 * Membuat tabel untuk data yang tidak dikenal
	 */
	private function createUnknownDataTable()
	{
		$sql = "CREATE TABLE IF NOT EXISTS unknown_import_data (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    baris_number INT,
                    kriteria TEXT,
                    sample TEXT,
                    created_by INT,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    INDEX idx_created_at (created_at)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

		$stmt = $this->db->prepare($sql);
		$stmt->execute();
	}

	/**
	 * Mendapatkan daftar audit yang aktif
	 */
	public function getDaftarAudit()
	{
		$sql = "SELECT id, nama_audit, tahun_audit, periode_audit, status
                FROM audit
                WHERE status IN ('draft', 'aktif')
                ORDER BY tahun_audit DESC, nama_audit ASC";

		$stmt = $this->db->prepare($sql);
		$stmt->execute();

		return $stmt->fetchAll(PDO::FETCH_ASSOC);
	}
}

// Proses request
if ($_SERVER["REQUEST_METHOD"] === "POST") {
	$import = new ImportAuditBerkas($pdo);

	if (isset($_POST["action"]) && $_POST["action"] === "get_audit_list") {
		// Return daftar audit dalam format JSON
		header("Content-Type: application/json");
		echo json_encode([
			"status" => "success",
			"data" => $import->getDaftarAudit(),
		]);
		exit();
	}

	if (isset($_POST["action"]) && $_POST["action"] === "import_excel") {
		$auditId = $_POST["audit_id"] ?? null;
		$userId = $_SESSION["user_id"] ?? 1; // Default user ID jika tidak ada session

		if (!$auditId) {
			echo json_encode([
				"status" => "error",
				"message" => "Audit ID harus dipilih",
			]);
			exit();
		}

		if (
			isset($_FILES["excel_file"]) &&
			$_FILES["excel_file"]["error"] === UPLOAD_ERR_OK
		) {
			$fileTmpPath = $_FILES["excel_file"]["tmp_name"];
			$fileName = $_FILES["excel_file"]["name"];
			$fileExtension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));

			// Validasi file extension
			if (!in_array($fileExtension, ["xlsx", "xls", "csv"])) {
				echo json_encode([
					"status" => "error",
					"message" =>
						"Format file tidak didukung. Gunakan file Excel (.xlsx, .xls) atau CSV",
				]);
				exit();
			}

			// Proses import
			$result = $import->importFromExcel($fileTmpPath, $auditId, $userId);

			echo json_encode($result);
			exit();
		} else {
			echo json_encode([
				"status" => "error",
				"message" => "File tidak berhasil diupload",
			]);
			exit();
		}
	}
}
?>
